/**
 * Cynthia Browser Extension - Background Service Worker
 * 
 * FULL-FEATURED browser automation supporting:
 * - Multi-tab management (list, switch, create, close)
 * - Navigation (url, back, forward, reload)
 * - Element interaction (click, type, fill, hover, drag, select)
 * - Keyboard (press keys, shortcuts)
 * - Data extraction (text, attributes, tables, lists)
 * - Waiting (for text, element, time)
 * - Screenshots and snapshots
 * - JavaScript evaluation
 */

const RELAY_URL = 'wss://relay.cynthiaconcierge.com';

let ws = null;
let connected = false;
let authenticated = false;
let userPhone = null;
let reconnectAttempts = 0;
const MAX_RECONNECT_ATTEMPTS = 50;

// Track element refs across tabs
const tabElements = new Map(); // tabId -> Map<ref, elementInfo>

function log(...args) {
  console.log('[Cynthia]', new Date().toISOString().substr(11, 8), ...args);
}

// ============ CONNECTION MANAGEMENT ============

chrome.runtime.onInstalled.addListener(() => {
  log('Extension installed');
  initializeConnection();
});

chrome.runtime.onStartup.addListener(() => {
  initializeConnection();
});

initializeConnection();

function initializeConnection() {
  chrome.storage.local.get(['phone'], (result) => {
    if (result.phone) {
      userPhone = result.phone;
      connect();
    }
  });
}

function connect() {
  if (ws && (ws.readyState === WebSocket.OPEN || ws.readyState === WebSocket.CONNECTING)) {
    return;
  }

  log('Connecting to relay...');

  try {
    ws = new WebSocket(RELAY_URL);
  } catch (err) {
    log('WebSocket creation failed:', err);
    scheduleReconnect();
    return;
  }

  ws.onopen = () => {
    log('Connected to relay');
    connected = true;
    reconnectAttempts = 0;
    updateBadge();

    if (userPhone) {
      ws.send(JSON.stringify({ type: 'auth', phone: userPhone }));
    }

    startKeepAlive();
  };

  ws.onmessage = async (event) => {
    try {
      const msg = JSON.parse(event.data);
      await handleMessage(msg);
    } catch (err) {
      log('Error handling message:', err);
    }
  };

  ws.onclose = () => {
    log('Disconnected from relay');
    connected = false;
    authenticated = false;
    ws = null;
    updateBadge();
    scheduleReconnect();
  };

  ws.onerror = (err) => {
    log('WebSocket error:', err);
  };
}

function scheduleReconnect() {
  if (reconnectAttempts >= MAX_RECONNECT_ATTEMPTS) {
    log('Max reconnect attempts reached');
    return;
  }
  
  reconnectAttempts++;
  const delay = Math.min(1000 * Math.pow(1.5, reconnectAttempts), 30000);
  
  setTimeout(() => {
    if (userPhone && (!ws || ws.readyState === WebSocket.CLOSED)) {
      connect();
    }
  }, delay);
}

function startKeepAlive() {
  chrome.alarms.clear('keepAlive');
  chrome.alarms.create('keepAlive', { periodInMinutes: 0.33 });
}

chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'keepAlive') {
    if (ws && ws.readyState === WebSocket.OPEN) {
      ws.send(JSON.stringify({ type: 'ping' }));
    } else if (userPhone) {
      connect();
    }
  }
});

// ============ MESSAGE HANDLING ============

async function handleMessage(msg) {
  switch (msg.type) {
    case 'welcome':
      log('Received welcome');
      break;
    case 'authenticated':
      log('Authenticated as:', msg.userId);
      authenticated = true;
      updateBadge();
      sendTabInfo();
      break;
    case 'error':
      log('Server error:', msg.message);
      break;
    case 'pong':
      break;
    case 'command':
      await executeCommand(msg);
      break;
  }
}

// ============ COMMAND EXECUTION ============

async function executeCommand(msg) {
  const { commandId, action, params } = msg;
  log('Command:', action, JSON.stringify(params).substring(0, 100));

  try {
    let result;

    switch (action) {
      // Tab management
      case 'tabs':
        result = await handleTabs(params);
        break;

      // Navigation
      case 'navigate':
        result = await navigateTo(params);
        break;
      case 'back':
        result = await goBack(params);
        break;
      case 'forward':
        result = await goForward(params);
        break;
      case 'reload':
        result = await reloadPage(params);
        break;

      // Snapshots
      case 'snapshot':
        result = await getPageSnapshot(params);
        break;
      case 'screenshot':
        result = await takeScreenshot(params);
        break;

      // Element interactions
      case 'click':
        result = await performClick(params);
        break;
      case 'type':
        result = await performType(params);
        break;
      case 'fill':
        result = await performFill(params);
        break;
      case 'hover':
        result = await performHover(params);
        break;
      case 'focus':
        result = await performFocus(params);
        break;
      case 'select':
        result = await performSelect(params);
        break;
      case 'drag':
        result = await performDrag(params);
        break;

      // Keyboard
      case 'press':
        result = await pressKey(params);
        break;

      // Scrolling
      case 'scroll':
        result = await performScroll(params);
        break;

      // Waiting
      case 'wait':
        result = await performWait(params);
        break;

      // Data extraction
      case 'getText':
        result = await getText(params);
        break;
      case 'getAttribute':
        result = await getAttribute(params);
        break;
      case 'getValue':
        result = await getValue(params);
        break;
      case 'extract':
        result = await extractData(params);
        break;

      // JavaScript
      case 'evaluate':
        result = await evaluateScript(params);
        break;

      default:
        result = { success: false, error: `Unknown action: ${action}` };
    }

    sendResponse(commandId, result);
  } catch (err) {
    log('Command error:', err);
    sendResponse(commandId, { success: false, error: err.message });
  }
}

function sendResponse(commandId, result) {
  if (ws && ws.readyState === WebSocket.OPEN) {
    const msgType = result.screenshot ? 'screenshot' : 
                    result.elements ? 'snapshot' : 'action_result';
    ws.send(JSON.stringify({ type: msgType, commandId, data: result, result }));
  }
}

// ============ TAB MANAGEMENT ============

async function handleTabs(params) {
  const action = params.action || 'list';

  switch (action) {
    case 'list': {
      const tabs = await chrome.tabs.query({ currentWindow: true });
      return {
        success: true,
        tabs: tabs.map((t, i) => ({
          index: i,
          id: t.id,
          url: t.url,
          title: t.title,
          active: t.active
        }))
      };
    }

    case 'select': {
      const tabs = await chrome.tabs.query({ currentWindow: true });
      const tab = tabs[params.index];
      if (!tab) return { success: false, error: `No tab at index ${params.index}` };
      await chrome.tabs.update(tab.id, { active: true });
      return { success: true, selected: tab.title };
    }

    case 'new': {
      const tab = await chrome.tabs.create({ url: params.url || 'about:blank' });
      return { success: true, tabId: tab.id };
    }

    case 'close': {
      if (params.index !== undefined) {
        const tabs = await chrome.tabs.query({ currentWindow: true });
        const tab = tabs[params.index];
        if (tab) await chrome.tabs.remove(tab.id);
      } else {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab) await chrome.tabs.remove(tab.id);
      }
      return { success: true };
    }

    default:
      return { success: false, error: `Unknown tabs action: ${action}` };
  }
}

// ============ NAVIGATION ============

async function navigateTo(params) {
  const tab = await getTargetTab(params.tabId);
  await chrome.tabs.update(tab.id, { url: params.url });
  return { success: true };
}

async function goBack(params) {
  const tab = await getTargetTab(params.tabId);
  await chrome.tabs.goBack(tab.id);
  return { success: true };
}

async function goForward(params) {
  const tab = await getTargetTab(params.tabId);
  await chrome.tabs.goForward(tab.id);
  return { success: true };
}

async function reloadPage(params) {
  const tab = await getTargetTab(params.tabId);
  await chrome.tabs.reload(tab.id);
  return { success: true };
}

// ============ SNAPSHOT ============

async function getPageSnapshot(params = {}) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (options) => {
      const elements = [];
      let refCounter = 0;
      const processed = new WeakSet();

      function isVisible(el) {
        if (!el.getBoundingClientRect) return false;
        const rect = el.getBoundingClientRect();
        if (rect.width === 0 || rect.height === 0) return false;
        const style = window.getComputedStyle(el);
        return style.display !== 'none' && 
               style.visibility !== 'hidden' && 
               parseFloat(style.opacity) > 0;
      }

      function isInteractive(el) {
        const tag = el.tagName?.toLowerCase();
        if (!tag) return false;

        // Always interactive
        if (['a', 'button', 'input', 'select', 'textarea', 'label', 'option'].includes(tag)) {
          return true;
        }

        // Event handlers
        if (el.onclick || el.onmousedown || el.onmouseup || el.onkeydown) return true;
        
        // ARIA roles
        const role = el.getAttribute('role');
        if (['button', 'link', 'textbox', 'checkbox', 'radio', 'menuitem', 'menuitemcheckbox',
             'menuitemradio', 'tab', 'switch', 'option', 'combobox', 'listbox', 'searchbox',
             'slider', 'spinbutton', 'scrollbar', 'separator', 'treeitem', 'gridcell'].includes(role)) {
          return true;
        }

        // Contenteditable
        if (el.contentEditable === 'true' || el.isContentEditable) return true;

        // Tabindex
        if (el.hasAttribute('tabindex') && el.tabIndex >= 0) return true;

        // Clickable cursor
        const style = window.getComputedStyle(el);
        if (style.cursor === 'pointer' || style.cursor === 'text') return true;

        // FACEBOOK-SPECIFIC: Look for form-like structures
        // Check if this div looks like an input wrapper (has a visible border, contains an input, or looks textbox-like)
        if (tag === 'div' || tag === 'span') {
          // Has a border that looks like an input
          if (style.borderWidth && parseFloat(style.borderWidth) >= 1) {
            const rect = el.getBoundingClientRect();
            // Reasonable input size
            if (rect.width > 100 && rect.height > 20 && rect.height < 100) {
              // Contains an input or is near form labels
              const hasInput = el.querySelector('input, textarea');
              const prevText = el.previousElementSibling?.innerText?.toLowerCase() || '';
              if (hasInput || prevText.includes('name') || prevText.includes('email') || prevText.includes('app')) {
                return true;
              }
            }
          }
        }

        // Common patterns
        const className = el.className?.toString?.() || '';
        if (/\b(btn|button|clickable|interactive|input|field|editable|link|action|trigger|_input|textinput)\b/i.test(className)) {
          return true;
        }

        // Data attributes (Facebook uses data-testid extensively)
        if (el.dataset && (el.dataset.testid || el.dataset.action || el.dataset.click || 
            el.dataset.toggle || el.dataset.target)) {
          return true;
        }

        // aria-* attributes that suggest interactivity
        if (el.hasAttribute('aria-haspopup') || el.hasAttribute('aria-expanded') || 
            el.hasAttribute('aria-pressed') || el.hasAttribute('aria-describedby')) {
          return true;
        }

        return false;
      }

      function getDescription(el) {
        const tag = el.tagName?.toLowerCase();
        
        // Inputs
        if (tag === 'input' || tag === 'textarea') {
          const type = el.type || 'text';
          const label = el.labels?.[0]?.textContent?.trim();
          return label || el.placeholder || el.getAttribute('aria-label') || el.name || `[${type} input]`;
        }

        // Select
        if (tag === 'select') {
          const label = el.labels?.[0]?.textContent?.trim();
          const selected = el.options[el.selectedIndex]?.text;
          return label || selected || el.getAttribute('aria-label') || '[dropdown]';
        }

        // Buttons/links
        if (['button', 'a'].includes(tag)) {
          return el.innerText?.trim()?.substring(0, 100) || el.getAttribute('aria-label') || el.title || '';
        }

        // Contenteditable
        if (el.isContentEditable) {
          return el.innerText?.trim()?.substring(0, 100) || el.getAttribute('aria-label') || '[editable]';
        }

        return (el.innerText || el.textContent || '')?.trim()?.substring(0, 100) || 
               el.getAttribute('aria-label') || el.title || '';
      }

      function processElement(el, depth = 0) {
        if (depth > 30) return;
        if (!el || processed.has(el)) return;
        processed.add(el);

        const tag = el.tagName?.toLowerCase();
        if (!tag) return;
        if (['script', 'style', 'noscript', 'svg', 'path', 'meta', 'link'].includes(tag)) return;
        
        // Check visibility (but still process children)
        const visible = isVisible(el);
        
        if (visible && (isInteractive(el) || !options.interactive)) {
          const ref = `e${refCounter++}`;
          const rect = el.getBoundingClientRect();
          
          const info = {
            tag,
            ref,
            type: el.type || null,
            text: getDescription(el),
            editable: el.isContentEditable || ['input', 'textarea'].includes(tag),
            checked: el.checked,
            disabled: el.disabled,
            visible: true,
            rect: {
              x: Math.round(rect.x),
              y: Math.round(rect.y),
              width: Math.round(rect.width),
              height: Math.round(rect.height)
            },
            attrs: {}
          };

          // Capture useful attributes
          ['id', 'name', 'class', 'href', 'type', 'placeholder', 'aria-label', 
           'role', 'data-testid', 'value', 'disabled', 'readonly', 'required',
           'src', 'alt', 'title'].forEach(attr => {
            let val = el.getAttribute(attr);
            if (val) {
              if (attr === 'class') val = val.substring(0, 100);
              if (attr === 'href' && val.length > 200) val = val.substring(0, 200) + '...';
              info.attrs[attr] = val;
            }
          });

          // For select elements, include options
          if (tag === 'select') {
            info.options = Array.from(el.options).slice(0, 20).map(o => ({
              value: o.value,
              text: o.text,
              selected: o.selected
            }));
          }

          el.dataset.cynthiaRef = ref;
          elements.push(info);
        }

        // Process children
        for (const child of el.children) {
          processElement(child, depth + 1);
        }
      }

      processElement(document.body);

      return {
        url: window.location.href,
        title: document.title,
        viewport: { width: window.innerWidth, height: window.innerHeight },
        scroll: { x: window.scrollX, y: window.scrollY },
        elementCount: elements.length,
        elements: elements.slice(0, 500)
      };
    },
    args: [{ interactive: params.interactive !== false }]
  });

  const result = results[0]?.result;
  
  // Store elements for this tab
  if (result?.elements) {
    const elemMap = new Map();
    result.elements.forEach(el => elemMap.set(el.ref, el));
    tabElements.set(tab.id, elemMap);
  }

  return result;
}

// ============ SCREENSHOT ============

async function takeScreenshot(params = {}) {
  const dataUrl = await chrome.tabs.captureVisibleTab(null, { format: 'png' });
  return { screenshot: dataUrl };
}

// ============ ELEMENT INTERACTIONS ============

async function getTargetTab(tabId) {
  if (tabId) {
    return await chrome.tabs.get(tabId);
  }
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab) throw new Error('No active tab');
  return tab;
}

async function performClick(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      let el;
      let x, y;
      
      // Support clicking by coordinates (x, y) when ref is not provided
      if (params.x !== undefined && params.y !== undefined) {
        x = params.x;
        y = params.y;
        el = document.elementFromPoint(x, y);
        if (!el) return { success: false, error: `No element at coordinates (${x}, ${y})` };
      } else if (params.ref) {
        el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
        if (!el) return { success: false, error: `Element ${params.ref} not found` };
      } else {
        return { success: false, error: 'Provide --ref or --x and --y coordinates' };
      }

      el.scrollIntoView({ behavior: 'instant', block: 'center' });

      const rect = el.getBoundingClientRect();
      if (x === undefined) x = rect.x + (params.offsetX ?? rect.width / 2);
      if (y === undefined) y = rect.y + (params.offsetY ?? rect.height / 2);

      const eventInit = {
        bubbles: true,
        cancelable: true,
        view: window,
        clientX: x,
        clientY: y,
        screenX: x,
        screenY: y,
        button: params.button === 'right' ? 2 : params.button === 'middle' ? 1 : 0,
        buttons: 1
      };

      // Add modifiers
      if (params.modifiers) {
        eventInit.ctrlKey = params.modifiers.includes('Control') || params.modifiers.includes('Ctrl');
        eventInit.shiftKey = params.modifiers.includes('Shift');
        eventInit.altKey = params.modifiers.includes('Alt');
        eventInit.metaKey = params.modifiers.includes('Meta') || params.modifiers.includes('Cmd');
      }

      el.dispatchEvent(new MouseEvent('mouseenter', eventInit));
      el.dispatchEvent(new MouseEvent('mouseover', eventInit));
      el.dispatchEvent(new MouseEvent('mousedown', eventInit));
      el.focus();
      el.dispatchEvent(new MouseEvent('mouseup', eventInit));
      
      if (params.doubleClick) {
        el.dispatchEvent(new MouseEvent('click', eventInit));
        el.dispatchEvent(new MouseEvent('mousedown', eventInit));
        el.dispatchEvent(new MouseEvent('mouseup', eventInit));
        el.dispatchEvent(new MouseEvent('click', eventInit));
        el.dispatchEvent(new MouseEvent('dblclick', eventInit));
      } else {
        el.dispatchEvent(new MouseEvent('click', eventInit));
      }

      return { success: true };
    },
    args: [params]
  });

  return results[0]?.result;
}

async function performType(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      let el;
      if (params.ref) {
        el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
        if (!el) return { success: false, error: `Element ${params.ref} not found` };
        el.focus();
      } else {
        // Type into currently focused element
        el = document.activeElement;
        if (!el || el === document.body) {
          return { success: false, error: 'No element focused. Click on input first or provide --ref' };
        }
      }

      if (el.isContentEditable) {
        // For contenteditable, insert text at cursor
        document.execCommand('insertText', false, params.text);
      } else {
        // For inputs, append to value
        const nativeSetter = Object.getOwnPropertyDescriptor(
          el.tagName === 'TEXTAREA' ? HTMLTextAreaElement.prototype : HTMLInputElement.prototype,
          'value'
        )?.set;
        
        const newValue = (el.value || '') + params.text;
        if (nativeSetter) {
          nativeSetter.call(el, newValue);
        } else {
          el.value = newValue;
        }
      }

      el.dispatchEvent(new Event('input', { bubbles: true }));
      el.dispatchEvent(new Event('change', { bubbles: true }));

      return { success: true };
    },
    args: [params]
  });

  return results[0]?.result;
}

async function performFill(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      let el;
      if (params.ref) {
        el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
        if (!el) return { success: false, error: `Element ${params.ref} not found` };
        el.focus();
      } else {
        // Fill currently focused element
        el = document.activeElement;
        if (!el || el === document.body) {
          return { success: false, error: 'No element focused. Click on input first or provide --ref' };
        }
      }

      if (el.isContentEditable) {
        el.innerHTML = '';
        document.execCommand('insertText', false, params.text || params.value);
      } else {
        const nativeSetter = Object.getOwnPropertyDescriptor(
          el.tagName === 'TEXTAREA' ? HTMLTextAreaElement.prototype : HTMLInputElement.prototype,
          'value'
        )?.set;
        
        if (nativeSetter) {
          nativeSetter.call(el, params.text || params.value || '');
        } else {
          el.value = params.text || params.value || '';
        }
      }

      el.dispatchEvent(new Event('input', { bubbles: true }));
      el.dispatchEvent(new Event('change', { bubbles: true }));

      return { success: true };
    },
    args: [params]
  });

  return results[0]?.result;
}

async function performHover(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      const el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
      if (!el) return { success: false, error: `Element ${params.ref} not found` };

      el.scrollIntoView({ behavior: 'instant', block: 'center' });
      
      const rect = el.getBoundingClientRect();
      const eventInit = {
        bubbles: true,
        cancelable: true,
        view: window,
        clientX: rect.x + rect.width / 2,
        clientY: rect.y + rect.height / 2
      };

      el.dispatchEvent(new MouseEvent('mouseenter', eventInit));
      el.dispatchEvent(new MouseEvent('mouseover', eventInit));
      el.dispatchEvent(new MouseEvent('mousemove', eventInit));

      return { success: true };
    },
    args: [params]
  });

  return results[0]?.result;
}

async function performFocus(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      const el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
      if (!el) return { success: false, error: `Element ${params.ref} not found` };

      el.scrollIntoView({ behavior: 'instant', block: 'center' });
      el.focus();

      return { success: true };
    },
    args: [params]
  });

  return results[0]?.result;
}

async function performSelect(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      const el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
      if (!el) return { success: false, error: `Element ${params.ref} not found` };
      if (el.tagName !== 'SELECT') return { success: false, error: 'Not a select element' };

      const values = Array.isArray(params.values) ? params.values : [params.value];
      
      // Find matching options
      for (const option of el.options) {
        option.selected = values.some(v => 
          option.value === v || 
          option.text === v || 
          option.text.toLowerCase().includes(v.toLowerCase())
        );
      }

      el.dispatchEvent(new Event('change', { bubbles: true }));
      el.dispatchEvent(new Event('input', { bubbles: true }));

      return { success: true, selected: el.value };
    },
    args: [params]
  });

  return results[0]?.result;
}

async function performDrag(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      const source = document.querySelector(`[data-cynthia-ref="${params.sourceRef}"]`);
      if (!source) return { success: false, error: `Source element ${params.sourceRef} not found` };

      const sourceRect = source.getBoundingClientRect();
      const startX = sourceRect.x + sourceRect.width / 2;
      const startY = sourceRect.y + sourceRect.height / 2;

      let endX, endY;
      if (params.targetRef) {
        const target = document.querySelector(`[data-cynthia-ref="${params.targetRef}"]`);
        if (!target) return { success: false, error: `Target element ${params.targetRef} not found` };
        const targetRect = target.getBoundingClientRect();
        endX = targetRect.x + targetRect.width / 2;
        endY = targetRect.y + targetRect.height / 2;
      } else {
        endX = params.targetX ?? startX;
        endY = params.targetY ?? startY;
      }

      const dataTransfer = new DataTransfer();
      
      source.dispatchEvent(new DragEvent('dragstart', {
        bubbles: true, clientX: startX, clientY: startY, dataTransfer
      }));
      
      source.dispatchEvent(new DragEvent('drag', {
        bubbles: true, clientX: endX, clientY: endY, dataTransfer
      }));
      
      const dropTarget = document.elementFromPoint(endX, endY);
      if (dropTarget) {
        dropTarget.dispatchEvent(new DragEvent('dragover', {
          bubbles: true, cancelable: true, clientX: endX, clientY: endY, dataTransfer
        }));
        dropTarget.dispatchEvent(new DragEvent('drop', {
          bubbles: true, clientX: endX, clientY: endY, dataTransfer
        }));
      }
      
      source.dispatchEvent(new DragEvent('dragend', {
        bubbles: true, clientX: endX, clientY: endY, dataTransfer
      }));

      return { success: true };
    },
    args: [params]
  });

  return results[0]?.result;
}

// ============ KEYBOARD ============

async function pressKey(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      const key = params.key;
      const parts = key.split('+');
      const mainKey = parts.pop();
      const modifiers = parts.map(m => m.toLowerCase());

      const eventInit = {
        bubbles: true,
        cancelable: true,
        key: mainKey,
        code: mainKey.length === 1 ? `Key${mainKey.toUpperCase()}` : mainKey,
        ctrlKey: modifiers.some(m => ['control', 'ctrl'].includes(m)),
        shiftKey: modifiers.includes('shift'),
        altKey: modifiers.some(m => ['alt', 'option'].includes(m)),
        metaKey: modifiers.some(m => ['meta', 'cmd', 'command', 'win'].includes(m))
      };

      const target = document.activeElement || document.body;
      target.dispatchEvent(new KeyboardEvent('keydown', eventInit));
      target.dispatchEvent(new KeyboardEvent('keypress', eventInit));
      target.dispatchEvent(new KeyboardEvent('keyup', eventInit));

      return { success: true };
    },
    args: [params]
  });

  return results[0]?.result;
}

// ============ SCROLLING ============

async function performScroll(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      if (params.ref) {
        const el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
        if (el) {
          el.scrollIntoView({ behavior: 'smooth', block: params.block || 'center' });
          return { success: true };
        }
        return { success: false, error: `Element ${params.ref} not found` };
      }

      if (params.top !== undefined || params.left !== undefined) {
        window.scrollTo({
          top: params.top ?? window.scrollY,
          left: params.left ?? window.scrollX,
          behavior: 'smooth'
        });
      } else {
        window.scrollBy({
          top: params.y || 0,
          left: params.x || 0,
          behavior: 'smooth'
        });
      }

      return { success: true, scroll: { x: window.scrollX, y: window.scrollY } };
    },
    args: [params]
  });

  return results[0]?.result;
}

// ============ WAITING ============

async function performWait(params) {
  const tab = await getTargetTab(params.tabId);

  // Fixed time wait
  if (params.time) {
    await new Promise(r => setTimeout(r, params.time * 1000));
    return { success: true, waited: params.time };
  }

  // Wait for text
  if (params.text || params.textGone) {
    const timeout = params.timeout || 30000;
    const startTime = Date.now();
    
    while (Date.now() - startTime < timeout) {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: (text, gone) => {
          const found = document.body.innerText.includes(text || gone);
          return gone ? !found : found;
        },
        args: [params.text, params.textGone]
      });

      if (results[0]?.result) {
        return { success: true, found: params.text || `(text gone: ${params.textGone})` };
      }

      await new Promise(r => setTimeout(r, 500));
    }

    return { success: false, error: 'Timeout waiting for text' };
  }

  return { success: false, error: 'Specify time, text, or textGone' };
}

// ============ DATA EXTRACTION ============

async function getText(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      if (params.ref) {
        const el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
        if (!el) return { success: false, error: `Element ${params.ref} not found` };
        return { success: true, text: el.innerText || el.textContent };
      }
      
      if (params.selector) {
        const el = document.querySelector(params.selector);
        if (!el) return { success: false, error: `No element matches selector` };
        return { success: true, text: el.innerText || el.textContent };
      }

      return { success: true, text: document.body.innerText };
    },
    args: [params]
  });

  return results[0]?.result;
}

async function getAttribute(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      const el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
      if (!el) return { success: false, error: `Element ${params.ref} not found` };
      return { success: true, value: el.getAttribute(params.name) };
    },
    args: [params]
  });

  return results[0]?.result;
}

async function getValue(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      const el = document.querySelector(`[data-cynthia-ref="${params.ref}"]`);
      if (!el) return { success: false, error: `Element ${params.ref} not found` };
      return { success: true, value: el.value || el.innerText };
    },
    args: [params]
  });

  return results[0]?.result;
}

async function extractData(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (params) => {
      const data = [];
      
      // Extract table
      if (params.type === 'table' || params.selector?.includes('table')) {
        const table = document.querySelector(params.selector || 'table');
        if (table) {
          const rows = table.querySelectorAll('tr');
          rows.forEach(row => {
            const cells = row.querySelectorAll('td, th');
            data.push(Array.from(cells).map(c => c.innerText.trim()));
          });
          return { success: true, data, type: 'table' };
        }
      }

      // Extract list
      if (params.type === 'list' || params.selector?.match(/ul|ol|li/)) {
        const list = document.querySelector(params.selector || 'ul, ol');
        if (list) {
          list.querySelectorAll('li').forEach(li => {
            data.push(li.innerText.trim());
          });
          return { success: true, data, type: 'list' };
        }
      }

      // Extract links
      if (params.type === 'links') {
        document.querySelectorAll(params.selector || 'a[href]').forEach(a => {
          data.push({ text: a.innerText.trim(), href: a.href });
        });
        return { success: true, data, type: 'links' };
      }

      // Extract by selector
      if (params.selector) {
        document.querySelectorAll(params.selector).forEach(el => {
          data.push(el.innerText.trim());
        });
        return { success: true, data, type: 'elements' };
      }

      return { success: false, error: 'Specify type or selector' };
    },
    args: [params]
  });

  return results[0]?.result;
}

// ============ JAVASCRIPT EVALUATION ============

async function evaluateScript(params) {
  const tab = await getTargetTab(params.tabId);
  
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (script) => {
      try {
        const result = eval(script);
        return { success: true, result: JSON.parse(JSON.stringify(result)) };
      } catch (err) {
        return { success: false, error: err.message };
      }
    },
    args: [params.script]
  });

  return results[0]?.result;
}

// ============ TAB INFO ============

async function sendTabInfo() {
  if (!ws || ws.readyState !== WebSocket.OPEN) return;

  try {
    const tabs = await chrome.tabs.query({ currentWindow: true });
    const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    ws.send(JSON.stringify({
      type: 'tab_info',
      tab: activeTab ? {
        id: activeTab.id,
        url: activeTab.url,
        title: activeTab.title
      } : null,
      allTabs: tabs.map((t, i) => ({
        index: i,
        id: t.id,
        title: t.title?.substring(0, 50),
        active: t.active
      }))
    }));
  } catch (err) {
    log('Error sending tab info:', err);
  }
}

// ============ BADGE ============

function updateBadge() {
  if (authenticated) {
    chrome.action.setBadgeText({ text: 'ON' });
    chrome.action.setBadgeBackgroundColor({ color: '#10B981' });
  } else if (connected) {
    chrome.action.setBadgeText({ text: '...' });
    chrome.action.setBadgeBackgroundColor({ color: '#F59E0B' });
  } else {
    chrome.action.setBadgeText({ text: '' });
  }
}

// ============ EVENT LISTENERS ============

chrome.tabs.onActivated.addListener(() => sendTabInfo());
chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
  if (changeInfo.status === 'complete') sendTabInfo();
});

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.type === 'getStatus') {
    sendResponse({ connected, authenticated, phone: userPhone });
  } else if (msg.type === 'setPhone') {
    userPhone = msg.phone;
    chrome.storage.local.set({ phone: msg.phone });
    
    if (!connected) {
      connect();
    } else if (ws && ws.readyState === WebSocket.OPEN) {
      ws.send(JSON.stringify({ type: 'auth', phone: msg.phone }));
    }
    sendResponse({ success: true });
  } else if (msg.type === 'disconnect') {
    userPhone = null;
    authenticated = false;
    chrome.storage.local.remove(['phone']);
    chrome.alarms.clear('keepAlive');
    if (ws) ws.close();
    updateBadge();
    sendResponse({ success: true });
  }
  return true;
});
