// Cynthia Browser Extension - Popup Script

document.addEventListener('DOMContentLoaded', () => {
  const statusDot = document.getElementById('statusDot');
  const statusText = document.getElementById('statusText');
  const loginView = document.getElementById('loginView');
  const connectedView = document.getElementById('connectedView');
  const phoneInput = document.getElementById('phoneInput');
  const connectBtn = document.getElementById('connectBtn');
  const disconnectBtn = document.getElementById('disconnectBtn');
  const connectedPhone = document.getElementById('connectedPhone');

  // Format phone number as user types
  phoneInput.addEventListener('input', (e) => {
    let value = e.target.value.replace(/\D/g, '');
    
    // Format as +1 (XXX) XXX-XXXX
    if (value.length > 0) {
      if (value.length <= 1) {
        value = '+' + value;
      } else if (value.length <= 4) {
        value = '+' + value.slice(0, 1) + ' (' + value.slice(1);
      } else if (value.length <= 7) {
        value = '+' + value.slice(0, 1) + ' (' + value.slice(1, 4) + ') ' + value.slice(4);
      } else {
        value = '+' + value.slice(0, 1) + ' (' + value.slice(1, 4) + ') ' + value.slice(4, 7) + '-' + value.slice(7, 11);
      }
    }
    
    e.target.value = value;
  });

  // Get current status
  function updateUI() {
    chrome.runtime.sendMessage({ type: 'getStatus' }, (response) => {
      if (!response) return;

      if (response.authenticated) {
        statusDot.className = 'status-dot connected';
        statusText.textContent = 'Connected';
        loginView.classList.add('hidden');
        connectedView.classList.remove('hidden');
        connectedPhone.textContent = formatPhone(response.phone);
      } else if (response.connected) {
        statusDot.className = 'status-dot connecting';
        statusText.textContent = 'Connecting...';
        loginView.classList.remove('hidden');
        connectedView.classList.add('hidden');
      } else {
        statusDot.className = 'status-dot disconnected';
        statusText.textContent = 'Not connected';
        loginView.classList.remove('hidden');
        connectedView.classList.add('hidden');
      }

      // Pre-fill phone if we have it
      if (response.phone && !response.authenticated) {
        phoneInput.value = formatPhone(response.phone);
      }
    });
  }

  // Format phone for display
  function formatPhone(phone) {
    if (!phone) return '';
    const digits = phone.replace(/\D/g, '');
    if (digits.length === 11 && digits.startsWith('1')) {
      return `+1 (${digits.slice(1, 4)}) ${digits.slice(4, 7)}-${digits.slice(7)}`;
    } else if (digits.length === 10) {
      return `+1 (${digits.slice(0, 3)}) ${digits.slice(3, 6)}-${digits.slice(6)}`;
    }
    return phone;
  }

  // Connect button
  connectBtn.addEventListener('click', () => {
    const phone = phoneInput.value.replace(/\D/g, '');
    if (phone.length < 10) {
      phoneInput.style.borderColor = '#EF4444';
      phoneInput.focus();
      return;
    }

    phoneInput.style.borderColor = '#E5E7EB';
    
    // Normalize to +1XXXXXXXXXX format
    const normalizedPhone = phone.length === 10 ? '+1' + phone : '+' + phone;

    connectBtn.textContent = 'Connecting...';
    connectBtn.disabled = true;

    chrome.runtime.sendMessage({ type: 'setPhone', phone: normalizedPhone }, (response) => {
      if (response?.success) {
        // Poll for connection status
        let attempts = 0;
        const checkConnection = setInterval(() => {
          attempts++;
          chrome.runtime.sendMessage({ type: 'getStatus' }, (status) => {
            if (status?.authenticated) {
              clearInterval(checkConnection);
              updateUI();
            } else if (attempts > 10) {
              clearInterval(checkConnection);
              connectBtn.textContent = 'Connect to Cynthia';
              connectBtn.disabled = false;
              statusText.textContent = 'Connection failed - try again';
            }
          });
        }, 500);
      } else {
        connectBtn.textContent = 'Connect to Cynthia';
        connectBtn.disabled = false;
      }
    });
  });

  // Disconnect button
  disconnectBtn.addEventListener('click', () => {
    chrome.runtime.sendMessage({ type: 'disconnect' }, () => {
      phoneInput.value = '';
      updateUI();
    });
  });

  // Enter key on phone input
  phoneInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
      connectBtn.click();
    }
  });

  // Initial update and poll for status changes
  updateUI();
  setInterval(updateUI, 2000);
});
